<?php

namespace App\Http\Controllers\Api;

use App\Models\Brand;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Throwable;

class BrandController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Brand::query();

        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        $brands = $query->get();

        return response()->json([
            'status' => true,
            'data' => $brands
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:brands,name',
        ]);

        try {
            $brand = Brand::create([
                'name' => $request->name,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Brand created successfully.',
                'data' => $brand
            ], 201);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to create brand.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json([
                'status' => false,
                'message' => 'Brand not found.'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $brand
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json([
                'status' => false,
                'message' => 'Brand not found.'
            ], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255|unique:brands,name,' . $id,
        ]);

        try {
            $brand->update([
                'name' => $request->name,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Brand updated successfully.',
                'data' => $brand
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to update brand.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json([
                'status' => false,
                'message' => 'Brand not found.'
            ], 404);
        }

        try {
            $brand->delete();

            return response()->json([
                'status' => true,
                'message' => 'Brand deleted successfully.'
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete brand.',
                'error' => $th->getMessage()
            ], 500);
        }
    }
}
