<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Notifications\OrderCreatedNotification;
use App\Notifications\OrderApprovedNotification;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Log;
use Throwable;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        /** @var User|null $user */
        $user = auth()->user();
        $query = Order::with(['customer', 'employee.user', 'items.product']);

        // Apply hierarchical filtering for non-admin users
        /** @var User|null $user */
        if ($user && !$user->hasAnyRole(['admin', 'super-admin'])) {
            $employeeDetail = $user->employeeDetail;
            if ($employeeDetail) {
                $subordinateIds = $employeeDetail->getAllSubordinateIds();
                $query->whereIn('employee_id', $subordinateIds);
            } else {
                // If user doesn't have employeeDetail, they shouldn't see any orders
                $query->whereRaw('1 = 0');
            }
        }

        // Search by Order Number
        if ($request->has('search')) {
            $query->where('order_number', 'like', "%{$request->search}%");
        }

        // Filter by Status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // Filter by Date
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('order_date', [$request->start_date, $request->end_date]);
        }

        // Filter by Customer
        if ($request->has('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        $orders = $query->orderBy('order_date', 'desc')->get();

        return response()->json([
            'status' => true,
            'data' => $orders
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'employee_id' => 'nullable|exists:employee_details,id',
            'order_date' => 'required|date',
            'payment_method' => 'in:cash,credit,bank_transfer,cheque',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'in:percentage,fixed',
            'note' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.price_type' => 'in:tp,flat',
            'items.*.bonus_quantity' => 'nullable|integer|min:0',
        ]);

        DB::beginTransaction();

        try {
            // Calculate Totals
            $subtotal = 0;
            $itemsData = [];

            foreach ($request->items as $item) {
                $totalPrice = $item['quantity'] * $item['unit_price'];
                $subtotal += $totalPrice;

                $itemsData[] = [
                    'product_id' => $item['product_id'],
                    'price_type' => $item['price_type'] ?? 'tp',
                    'unit_price' => $item['unit_price'],
                    'quantity' => $item['quantity'],
                    'bonus_quantity' => $item['bonus_quantity'] ?? 0,
                    'total_price' => $totalPrice,
                ];
            }

            // Calculate Discount
            $discountAmount = 0;
            $discountValue = $request->discount ?? 0;
            if ($request->discount_type === 'percentage') {
                $discountAmount = ($subtotal * $discountValue) / 100;
            } else {
                $discountAmount = $discountValue;
            }

            $totalAmount = $subtotal - $discountAmount;

            // Generate Order Number
            $lastOrder = Order::latest()->first();
            $lastId = $lastOrder ? $lastOrder->id : 0;
            $orderNumber = 'ORD-' . str_pad($lastId + 1, 6, '0', STR_PAD_LEFT);

            // Create Order
            $order = Order::create([
                'order_number' => $orderNumber,
                'customer_id' => $request->customer_id,
                'employee_id' => $request->employee_id,
                'order_date' => $request->order_date,
                'payment_method' => $request->payment_method ?? 'cash',
                'subtotal' => $subtotal,
                'discount' => $discountValue,
                'discount_type' => $request->discount_type ?? 'percentage',
                'discount_amount' => $discountAmount,
                'total_amount' => $totalAmount,
                'status' => 'pending',
                'note' => $request->note,
                'created_by' => auth()->id(),
            ]);

            // Create Order Items
            foreach ($itemsData as $itemData) {
                // Here you could also decrement product stock if required immediately
                // For now sticking to CRUD creation logic
                $order->items()->create($itemData);
            }

            DB::commit();

            // Send Notification
            try {
                // Reload order with relations for notification
                $order->load(['employee', 'customer']);
                $employee = $order->employee;

                if ($employee && $employee->parent_id) {
                    $parentEmployee = $employee->parent;
                    if ($parentEmployee && $parentEmployee->user) {
                        $parentEmployee->user->notify(new OrderCreatedNotification($order));
                    }
                } else {
                    // Fallback to Admin and Super Admin
                    $admins = User::role(['admin', 'super-admin'])->get();
                    Notification::send($admins, new OrderCreatedNotification($order));
                }
            } catch (Throwable $notifError) {
                // We log notification errors but don't fail the order creation
                \Log::error('Order creation notification failed: ' . $notifError->getMessage());
            }

            return response()->json([
                'status' => true,
                'message' => 'Order created successfully.',
                'data' => $order->load('items')
            ], 201);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to create order.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $order = Order::with(['customer', 'employee', 'items.product'])->find($id);

        if (!$order) {
            return response()->json([
                'status' => false,
                'message' => 'Order not found.'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $order
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'status' => false,
                'message' => 'Order not found.'
            ], 404);
        }

        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'employee_id' => 'nullable|exists:employee_details,id',
            'order_date' => 'required|date',
            'payment_method' => 'in:cash,credit,bank_transfer,cheque',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'in:percentage,fixed',
            'note' => 'nullable|string',
            'status' => 'in:pending,approved,delivered,cancelled',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:order_items,id',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.price_type' => 'in:tp,flat',
            'items.*.bonus_quantity' => 'nullable|integer|min:0',
        ]);

        DB::beginTransaction();

        try {
            // Update basic order fields
            $order->customer_id = $request->customer_id;
            $order->employee_id = $request->employee_id;
            $order->order_date = $request->order_date;
            $order->payment_method = $request->payment_method ?? 'cash';
            $order->discount = $request->discount ?? 0;
            $order->discount_type = $request->discount_type ?? 'percentage';
            $order->status = $request->status ?? $order->status;
            $order->note = $request->note;

            // Sync Order Items
            $requestItemIds = collect($request->items)->pluck('id')->filter()->toArray();

            // Delete items not in the request
            OrderItem::where('order_id', $order->id)
                ->whereNotIn('id', $requestItemIds)
                ->delete();

            // Calculate Totals
            $subtotal = 0;

            foreach ($request->items as $item) {
                $totalPrice = $item['quantity'] * $item['unit_price'];
                $subtotal += $totalPrice;

                $itemData = [
                    'product_id' => $item['product_id'],
                    'price_type' => $item['price_type'] ?? 'tp',
                    'unit_price' => $item['unit_price'],
                    'quantity' => $item['quantity'],
                    'bonus_quantity' => $item['bonus_quantity'] ?? 0,
                    'total_price' => $totalPrice,
                ];

                if (isset($item['id'])) {
                    // Update existing item
                    OrderItem::where('id', $item['id'])
                        ->where('order_id', $order->id)
                        ->update($itemData);
                } else {
                    // Create new item
                    $order->items()->create($itemData);
                }
            }

            // Recalculate Discount and Total
            $discountAmount = 0;
            $discountValue = $order->discount;
            if ($order->discount_type === 'percentage') {
                $discountAmount = ($subtotal * $discountValue) / 100;
            } else {
                $discountAmount = $discountValue;
            }

            $totalAmount = $subtotal - $discountAmount;

            // Update totals
            $order->subtotal = $subtotal;
            $order->discount_amount = $discountAmount;
            $order->total_amount = $totalAmount;
            $order->save();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Order updated successfully.',
                'data' => $order->load('items.product', 'customer', 'employee')
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to update order.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'status' => false,
                'message' => 'Order not found.'
            ], 404);
        }

        try {
            // Delete order (Cascade deletes items)
            $order->delete();

            return response()->json([
                'status' => true,
                'message' => 'Order deleted successfully.'
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete order.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Approve an order.
     */
    public function approve($id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'status' => false,
                'message' => 'Order not found.'
            ], 404);
        }

        // Check if order is already approved or in a final state
        if ($order->status === 'approved') {
            return response()->json([
                'status' => false,
                'message' => 'Order is already approved.'
            ], 400);
        }

        if ($order->status === 'delivered') {
            return response()->json([
                'status' => false,
                'message' => 'Cannot approve a delivered order.'
            ], 400);
        }

        if ($order->status === 'cancelled') {
            return response()->json([
                'status' => false,
                'message' => 'Cannot approve a cancelled order.'
            ], 400);
        }

        try {
            $order->status = 'approved';
            $order->approved_by = auth()->id();
            $order->save();

            // Send Notification to the employee who created the order
            try {
                $order->load('employee.user');
                if ($order->employee && $order->employee->user) {
                    $order->employee->user->notify(new OrderApprovedNotification($order));
                }
            } catch (Throwable $notifError) {
                \Log::error('Order approval notification failed: ' . $notifError->getMessage());
            }

            return response()->json([
                'status' => true,
                'message' => 'Order approved successfully.',
                'data' => $order->load('items.product', 'customer', 'employee')
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to approve order.',
                'error' => $th->getMessage()
            ], 500);
        }
    }
}
