<?php

namespace App\Http\Controllers\Api;

use App\Models\Product;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Throwable;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Product::with(['category', 'brand']);

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('pack_size', 'like', "%{$search}%");
            });
        }

        // Filter by Category
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // Filter by Brand
        if ($request->has('brand_id')) {
            $query->where('brand_id', $request->brand_id);
        }

        $products = $query->get();

        return response()->json([
            'status' => true,
            'data' => $products
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'pack_size' => 'nullable|string|max:100',
            'purchase_price' => 'required|numeric|min:0',
            'sale_price' => 'required|numeric|min:0',
            'flat_price' => 'nullable|numeric|min:0',
            'quantity' => 'nullable|integer|min:0',
            'expiration_date' => 'nullable|date',
            'image' => 'nullable|string',
        ]);

        try {
            $imagePath = null;
            if ($request->image) {
                // Handle Base64 Image
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    $imageName = 'product_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('products/' . $imageName, $image);
                    $imagePath = 'products/' . $imageName;
                }
            }

            $product = Product::create([
                'category_id' => $request->category_id,
                'brand_id' => $request->brand_id,
                'name' => $request->name,
                'pack_size' => $request->pack_size,
                'purchase_price' => $request->purchase_price,
                'sale_price' => $request->sale_price,
                'flat_price' => $request->flat_price ?? 0,
                'quantity' => $request->quantity ?? 0,
                'expiration_date' => $request->expiration_date,
                'image' => $imagePath
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Product created successfully.',
                'data' => $product
            ], 201);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to create product.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $product = Product::with(['category', 'brand'])->find($id);

        if (!$product) {
            return response()->json([
                'status' => false,
                'message' => 'Product not found.'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $product
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $product = Product::find($id);

        if (!$product) {
            return response()->json([
                'status' => false,
                'message' => 'Product not found.'
            ], 404);
        }

        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|max:255',
            'pack_size' => 'nullable|string|max:100',
            'purchase_price' => 'required|numeric|min:0',
            'sale_price' => 'required|numeric|min:0',
            'flat_price' => 'nullable|numeric|min:0',
            'quantity' => 'nullable|integer|min:0',
            'expiration_date' => 'nullable|date',
            'image' => 'nullable|string',
        ]);

        try {
            if ($request->filled('image')) {
                // Handle Base64 Image
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    // Delete old image
                    if ($product->image && Storage::disk('public')->exists($product->image)) {
                        Storage::disk('public')->delete($product->image);
                    }

                    $imageName = 'product_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('products/' . $imageName, $image);
                    $product->image = 'products/' . $imageName;
                }
            }

            $product->update([
                'category_id' => $request->category_id,
                'brand_id' => $request->brand_id,
                'name' => $request->name,
                'pack_size' => $request->pack_size,
                'purchase_price' => $request->purchase_price,
                'sale_price' => $request->sale_price,
                'flat_price' => $request->flat_price ?? 0,
                // Quantity is typically not updated directly here to ensure history tracking, 
                // but usually allowed in basic CRUD. 
                // For strict stock management, quantity updates should go through stockIn/stockOut.
                // However, allowing it here for correction purposes.
                'quantity' => $request->quantity ?? $product->quantity,
                'expiration_date' => $request->expiration_date,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Product updated successfully.',
                'data' => $product
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to update product.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $product = Product::find($id);

        if (!$product) {
            return response()->json([
                'status' => false,
                'message' => 'Product not found.'
            ], 404);
        }

        try {
            // Delete Image
            if ($product->image && Storage::disk('public')->exists($product->image)) {
                Storage::disk('public')->delete($product->image);
            }

            $product->delete();

            return response()->json([
                'status' => true,
                'message' => 'Product deleted successfully.'
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete product.',
                'error' => $th->getMessage()
            ], 500);
        }
    }
}
