<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Stock;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Throwable;

class StockController extends Controller
{
    /**
     * Add Stock (Stock In)
     */
    public function stockIn(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string|max:255',
            'date' => 'required|date',
            'remarks' => 'nullable|string',
        ]);

        DB::beginTransaction();

        try {
            $product = Product::find($request->product_id);

            // Create Stock Log
            $stock = Stock::create([
                'product_id' => $request->product_id,
                'supplier_id' => $request->supplier_id,
                'type' => 'in',
                'quantity' => $request->quantity,
                'reason' => $request->reason,
                'date' => $request->date,
                'remarks' => $request->remarks,
            ]);

            // Update Product Quantity
            $product->quantity += $request->quantity;
            $product->save();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Stock added successfully.',
                'data' => [
                    'stock_log' => $stock,
                    'current_product_stock' => $product->quantity
                ]
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to add stock.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove Stock (Stock Out)
     */
    public function stockOut(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string|max:255',
            'date' => 'required|date',
            'remarks' => 'nullable|string',
        ]);

        DB::beginTransaction();

        try {
            $product = Product::find($request->product_id);

            // Check if sufficient stock is available
            if ($product->quantity < $request->quantity) {
                return response()->json([
                    'status' => false,
                    'message' => 'Insufficient stock.',
                    'current_stock' => $product->quantity
                ], 400);
            }

            // Create Stock Log
            $stock = Stock::create([
                'product_id' => $request->product_id,
                'type' => 'out',
                'quantity' => $request->quantity,
                'reason' => $request->reason,
                'date' => $request->date,
                'remarks' => $request->remarks,
            ]);

            // Update Product Quantity
            $product->quantity -= $request->quantity;
            $product->save();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Stock removed successfully.',
                'data' => [
                    'stock_log' => $stock,
                    'current_product_stock' => $product->quantity
                ]
            ]);
        } catch (Throwable $th) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Failed to remove stock.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Stock History (All)
     */
    public function history(Request $request)
    {
        $query = Stock::with(['product', 'supplier'])->orderBy('date', 'desc');

        // Filter by Product
        if ($request->has('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        // Filter by Type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by Date Range
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('date', [$request->start_date, $request->end_date]);
        }

        $history = $query->get();

        return response()->json([
            'status' => true,
            'data' => $history
        ]);
    }

    /**
     * Product Specific History
     */
    public function productHistory($id)
    {
        $history = Stock::with('supplier')
            ->where('product_id', $id)
            ->orderBy('date', 'desc')
            ->get();

        return response()->json([
            'status' => true,
            'data' => $history
        ]);
    }
}
