<?php

namespace App\Http\Controllers\Api;

use App\Models\Supplier;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Throwable;

class SupplierController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Supplier::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('company_name', 'like', "%{$search}%")
                    ->orWhere('proprietor_name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $suppliers = $query->get();

        return response()->json([
            'status' => true,
            'data' => $suppliers
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_name' => 'required|string|max:255',
            'proprietor_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'whatsapp' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'required|string',
            'country' => 'required|string|max:100',
            'status' => 'in:active,inactive',
            'image' => 'nullable|string',
        ]);

        try {
            $imagePath = null;
            if ($request->image) {
                // Handle Base64 Image
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    $imageName = 'supplier_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('suppliers/' . $imageName, $image);
                    $imagePath = 'suppliers/' . $imageName;
                }
            }

            $supplier = Supplier::create([
                'company_name' => $request->company_name,
                'proprietor_name' => $request->proprietor_name,
                'phone' => $request->phone,
                'whatsapp' => $request->whatsapp,
                'email' => $request->email,
                'address' => $request->address,
                'country' => $request->country,
                'status' => $request->status ?? 'active',
                'image' => $imagePath
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Supplier created successfully.',
                'data' => $supplier
            ], 201);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to create supplier.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $supplier = Supplier::find($id);

        if (!$supplier) {
            return response()->json([
                'status' => false,
                'message' => 'Supplier not found.'
            ], 404);
        }

        return response()->json([
            'status' => true,
            'data' => $supplier
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $supplier = Supplier::find($id);

        if (!$supplier) {
            return response()->json([
                'status' => false,
                'message' => 'Supplier not found.'
            ], 404);
        }

        $request->validate([
            'company_name' => 'required|string|max:255',
            'proprietor_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'whatsapp' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'required|string',
            'country' => 'required|string|max:100',
            'status' => 'in:active,inactive',
            'image' => 'nullable|string',
        ]);

        try {
            if ($request->filled('image')) {
                // Handle Base64 Image
                if (preg_match('/^data:image\/(\w+);base64,/', $request->image, $type)) {
                    $image = substr($request->image, strpos($request->image, ',') + 1);
                    $type = strtolower($type[1]); // jpg, png, gif

                    if (!in_array($type, ['jpg', 'jpeg', 'gif', 'png'])) {
                        throw new \Exception('Invalid image type');
                    }

                    $image = str_replace(' ', '+', $image);
                    $image = base64_decode($image);

                    if ($image === false) {
                        throw new \Exception('Base64 decode failed');
                    }

                    // Delete old image
                    if ($supplier->image && Storage::disk('public')->exists($supplier->image)) {
                        Storage::disk('public')->delete($supplier->image);
                    }

                    $imageName = 'supplier_' . time() . '_' . Str::random(10) . '.' . $type;
                    Storage::disk('public')->put('suppliers/' . $imageName, $image);
                    $supplier->image = 'suppliers/' . $imageName;
                }
            }

            $supplier->update([
                'company_name' => $request->company_name,
                'proprietor_name' => $request->proprietor_name,
                'phone' => $request->phone,
                'whatsapp' => $request->whatsapp,
                'email' => $request->email,
                'address' => $request->address,
                'country' => $request->country,
                'status' => $request->status ?? $supplier->status
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Supplier updated successfully.',
                'data' => $supplier
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to update supplier.',
                'error' => $th->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $supplier = Supplier::find($id);

        if (!$supplier) {
            return response()->json([
                'status' => false,
                'message' => 'Supplier not found.'
            ], 404);
        }

        try {
            // Delete Image
            if ($supplier->image && Storage::disk('public')->exists($supplier->image)) {
                Storage::disk('public')->delete($supplier->image);
            }

            $supplier->delete();

            return response()->json([
                'status' => true,
                'message' => 'Supplier deleted successfully.'
            ]);
        } catch (Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Failed to delete supplier.',
                'error' => $th->getMessage()
            ], 500);
        }
    }
}
